<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Project;
use App\Models\Campaign;
use App\Models\MobileNotification;
use App\Models\MobileBanner;
use App\Models\MobileCaseApplication;
use App\Models\MobileInKindDonation;
use Illuminate\Support\Facades\Storage;

class MobileContentController extends Controller
{
    // --- Dashboard View to Manage All Mobile Content ---
    public function index()
    {
        $projects = Project::where('show_on_mobile', true)->get();
        $campaigns = Campaign::where('show_on_mobile', true)->get();
        return view('mobile.dashboard', compact('projects', 'campaigns'));
    }

    // --- Banners Management ---
    public function bannersIndex()
    {
        $banners = MobileBanner::orderBy('sort_order')->get();
        return view('mobile.banners', compact('banners'));
    }

    public function bannerStore(Request $request)
    {
        $data = $request->validate([
            'image' => 'required|image|max:5120',
            'title' => 'nullable|string',
            'link_type' => 'nullable|string',
            'link_id' => 'nullable|string',
            'sort_order' => 'integer'
        ]);

        if ($request->hasFile('image')) {
            $data['image_path'] = $request->file('image')->store('mobile/banners', 'public');
        }

        MobileBanner::create($data);
        return back()->with('success', 'Banner created successfully');
    }

    public function bannerDestroy(MobileBanner $banner)
    {
        if ($banner->image_path) Storage::disk('public')->delete($banner->image_path);
        $banner->delete();
        return back()->with('success', 'Banner deleted successfully');
    }

    // --- Notifications Management ---
    public function notificationsIndex()
    {
        $notifications = MobileNotification::orderByDesc('created_at')->get();
        return view('mobile.notifications', compact('notifications'));
    }

    public function notificationStore(Request $request)
    {
        $data = $request->validate([
            'title' => 'required|string',
            'body' => 'required|string',
            'target_audience' => 'nullable|string',
            'image' => 'nullable|image'
        ]);

        if ($request->hasFile('image')) {
            $data['image_path'] = $request->file('image')->store('mobile/notifications', 'public');
        }

        // Logic to send push notification via Firebase/OneSignal would go here
        $data['is_sent'] = true;
        $data['sent_at'] = now();

        MobileNotification::create($data);
        return back()->with('success', 'Notification sent successfully');
    }

    // --- Case Applications (e.g. needy cases applying) ---
    public function casesIndex()
    {
        $applications = MobileCaseApplication::orderByDesc('created_at')->get();
        return view('mobile.cases', compact('applications'));
    }

    public function caseUpdateStatus(Request $request, MobileCaseApplication $application)
    {
        $application->update(['status' => $request->status, 'admin_notes' => $request->admin_notes]);
        return back()->with('success', 'Case status updated');
    }

    // --- In-Kind Donations (e.g. clothes, furniture) ---
    public function inKindDonationsIndex()
    {
        $donations = MobileInKindDonation::orderByDesc('created_at')->get();
        return view('mobile.in_kind_donations', compact('donations'));
    }

    public function inKindDonationUpdateStatus(Request $request, MobileInKindDonation $donation)
    {
        $donation->update(['status' => $request->status]);
        return back()->with('success', 'Donation status updated');
    }

    // --- Update Mobile Specific Contentfor Projects/Campaigns ---
    public function updateProjectMobileContent(Request $request, Project $project)
    {
        $project->update([
            'mobile_content' => $request->mobile_content,
            'show_on_mobile' => $request->has('show_on_mobile')
        ]);
        return back()->with('success', 'Project mobile content updated');
    }

    public function updateCampaignMobileContent(Request $request, Campaign $campaign)
    {
        $campaign->update([
            'mobile_content' => $request->mobile_content,
            'show_on_mobile' => $request->has('show_on_mobile')
        ]);
        return back()->with('success', 'Campaign mobile content updated');
    }
}
